#!/bin/bash
#
# Copyright (C) 2023 MOXA Inc. All rights reserved.
# This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
# See the file LICENSE for details.
#
# Authors:
#       2023  Wilson YS Huang  <wilsonys.huang@moxa.com>

# shellcheck disable=SC2317 # Don't warn about unreachable commands in this file

source "${INSTL_SCRIPT_DIR}/lib/util.sh"
source "${INSTL_SCRIPT_DIR}/lib/logger.sh"
log::init "${INSTL_LOG_FILE}" "${INSTL_LOG_LEVEL}"

set -e

process_file_test() {
        local repo_name="$1"
        local repo_tag="$2"
        local repo_ver

        if util::is_exists "${INSTL_SRC_VERSION_FILE}" && grep -wq "${repo_name}" "${INSTL_SRC_VERSION_FILE}"; then
                repo_ver=$(grep -w "${repo_name}" "${INSTL_SRC_VERSION_FILE}" | awk -F ':' '{print $2}')
        else
                repo_ver="N/A"
        fi

        log::info "$(printf " %-50s%-15s%-15s%-25s\n" "${repo_name}" "" "" "${repo_ver}")"

        source "${INSTL_SCRIPT_DIR}/self-test.d/${repo_name}.test.sh" "${repo_tag}"

        return 0
}

process_config_test() {
        local src_name="$1"
        local dst_path="$3"
        local verify_path

        if util::is_dir "${dst_path}"; then
                verify_path="${dst_path}${src_name}"
        else
                verify_path="${dst_path}"
        fi

        if util::is_exists "${verify_path}"; then
                log::info "$(printf " %-50s$(colors::green %-15s)%-15s%-25s\n" "- ${src_name}" "Yes" "N/A" "N/A")"
        else
                log::info "$(printf " %-50s$(colors::red %-15s)%-15s%-25s\n" "- ${src_name}" "No" "N/A" "N/A")"
        fi

        return 0
}

process_package_test() {
        local pkg_name="$1"
        local pkg_ver="$2"
        local ret_val=1

        log::info "$(printf " %-50s$(colors::green %-15s)%-15s%-25s\n" "${pkg_name}" "" "" "")"

        find "${INSTL_TOP_DIR}/pkg/${pkg_name}/${pkg_ver}" -maxdepth 1 -type f -name '*.deb' -print0 |
                while IFS= read -r -d '' pkg_file; do
                        binary_pkg_name=$(dpkg-deb --field "${pkg_file}" package)
                        binary_pkg_version=$(dpkg-deb --field "${pkg_file}" version)
                        dpkg-query -W -f='${Package}, ${Version}\n' "${binary_pkg_name}" 2>/dev/null | grep -q "${binary_pkg_version}"
                        ret_val=${?}

                        if [[ $ret_val -eq 0 ]]; then
                                log::info "$(printf " %-50s$(colors::green %-15s)%-15s%-25s\n" "- ${binary_pkg_name}" "Yes" "N/A" "${binary_pkg_version}")"
                        else
                                log::info "$(printf " %-50s$(colors::red %-15s)%-15s%-25s\n" "- ${binary_pkg_name}" "No" "N/A" "${binary_pkg_version}")"
                        fi
                done
}

main() {
        local drivers_list
        local tools_list
        local driver_available
        local tool_available

        driver_available=true
        tool_available=true

        log::info "$(printf "%105s" " " | tr ' ' '-')"
        log::info "$(printf " %-50s%-15s%-15s%-25s\n" "Name" "Installed" "Status" "Version")"
        log::info "$(printf "%105s" " " | tr ' ' '=')"

        drivers_list="${INSTL_PRODUCT_DIR}/${BOARD_ID}/drivers"
        if ! util::parse_product_file "${drivers_list}" "${LINUX_KERNEL_MAJOR_VER}" "${LINUX_KERNEL_MINOR_VER}" process_file_test; then
                driver_available=false
        fi

        tools_list="${INSTL_PRODUCT_DIR}/${BOARD_ID}/tools"
        if ! util::parse_product_file "${tools_list}" "${LINUX_KERNEL_MAJOR_VER}" "${LINUX_KERNEL_MINOR_VER}" process_file_test; then
                tool_available=false
        fi

        if [[ $driver_available = false ]] && [[ $tool_available = false ]]; then
                log::info "$(printf " %-50s%-15s%-15s%-25s\n" "No available" "N/A" "N/A" "N/A")"
        fi

        if [[ $INSTL_VERBOSE = true ]]; then
                local file_available
                local pkg_available
                file_available=true
                pkg_available=true

                if [ -n "${LINUX_DISTRO_VERSION_CODENAME}" ]; then
                        codename="${LINUX_DISTRO_VERSION_CODENAME}"
                elif [ "${LINUX_DISTRO_ID}" = "rhel" ] || [ "${LINUX_DISTRO_ID}" = "rocky" ]; then
                        codename="${LINUX_DISTRO_ID}_$(echo "${LINUX_DISTRO_VERSION}" | sed -n 's/.*(\(.*\)).*/\1/p')"
                else
                        codename="${LINUX_DISTRO_ID}_${LINUX_DISTRO_VERSION_ID}"
                fi

                files_list="${INSTL_PRODUCT_DIR}/${BOARD_ID}/files"
                if util::is_exists "${files_list}"; then
                        log::info "$(printf " %-50s%-15s%-15s%-25s\n" "Config files" "" "" "")"
                        if ! util::parse_product_file_from_codename "${files_list}" "${codename}" process_config_test; then
                                file_available=false
                        fi

                        if [[ $file_available = false ]]; then
                                log::info "$(printf " %-50s%-15s%-15s%-25s\n" "No available" "N/A" "N/A" "N/A")"
                        fi
                fi

                pkg_list="${INSTL_PRODUCT_DIR}/${BOARD_ID}/packages"
                if util::is_exists "${pkg_list}"; then
                        if ! util::parse_product_file_from_codename "${pkg_list}" "${codename}" process_package_test; then
                                pkg_available=false
                        fi

                        if [[ $pkg_available = false ]]; then
                                log::info "$(printf " %-50s%-15s%-15s%-25s\n" "No available" "N/A" "N/A" "N/A")"
                        fi
                fi
        fi

        log::info "$(printf "%105s" " " | tr ' ' '-')"

        exit 0
}

main
