/*
 * Copyright (C) MOXA Inc. All rights reserved.
 * Authors:
 *     2024  Wilson YS Huang  <wilsonys.huang@moxa.com>
 * This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
 * See the file LICENSE for details.
 */

#include <errno.h>
#include <signal.h>
#include <stdint.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <sys/poll.h>
#include <sys/socket.h>
#include <termios.h>
#include <time.h>
#include <unistd.h>

#include <common.h>
#include <logger.h>
#include <mcu.h>
#include <socket.h>
#include <tty.h>

typedef struct {
    const char *model_name;
    const char *mcu_tty_device;
    size_t      name_prefix_sz;
    int32_t     mcu_features;
} model_information;

typedef void (*model_init_func)(const model_information *info);

typedef struct {
    model_information info;
    model_init_func   init_func;
} model_t;

static bool mcu_capability = true;

static void V3000_init_func(const model_information *info)
{
    const uint32_t IO_BOARD_GPIO_PIN = 47;
    const uint32_t IO_BOARD_4LAN     = 1;
    const uint32_t IO_BOARD_8LAN     = 0;
    int32_t        gpio_val          = -1;
    int32_t        gpio_base         = -1;

    (void)info;

    if (!get_gpio_base_by_label("gpio_it87", &gpio_base)) {
        log_error("Get gpio_it87 base failed");
        return;
    }

    if (!get_gpio_value(gpio_base + IT87_GPIO_MAPPING(IO_BOARD_GPIO_PIN), &gpio_val)) {
        log_error("Get IO board gpio value failed");
        return;
    }

    if (gpio_val == IO_BOARD_4LAN) {
        mcu_capability = false;
        log_warn("MCU is not available on 4 LAN IO board");
    }
    else if (gpio_val == IO_BOARD_8LAN) {
        mcu_capability = true;
    }
    else {
        log_error("Unknown IO board");
    }
}

static const model_t support_models[] = {
    {.info = {
         .model_name     = "V3000",   // In fact, it is V3200 series.
         .mcu_tty_device = "/dev/ttyS2",
         .name_prefix_sz = 5,
         .mcu_features   = FEAT_MCU_VERSION | FEAT_MCU_RELAY_MODE | FEAT_MCU_WDT_RELAY_MODE | FEAT_MCU_POWEROFF_RELAY_MODE | FEAT_MCU_APP_WDT_RELAY_MODE | FEAT_MCU_APP_WDT_TIMEOUT_CMD,
     },
     .init_func = V3000_init_func},
    {.info = {
         .model_name     = "V3400",
         .mcu_tty_device = "/dev/ttyS2",
         .name_prefix_sz = 5,
         .mcu_features   = FEAT_MCU_VERSION | FEAT_MCU_RELAY_MODE | FEAT_MCU_WDT_RELAY_MODE | FEAT_MCU_POWEROFF_RELAY_MODE | FEAT_MCU_APP_WDT_RELAY_MODE | FEAT_MCU_APP_WDT_TIMEOUT_CMD,
     },
     .init_func = V3000_init_func},
    {{NULL, NULL, 0, 0}, NULL}};

static const model_t *current_model = NULL;

typedef struct proxy_ctx {
    int32_t un_skt_server_fd;
    int32_t mcu_tty_fd;
} proxy_ctx;

proxy_ctx *ctx = NULL;

static proxy_ctx *proxy_new()
{
    proxy_ctx *ctx = calloc(1, sizeof(proxy_ctx));
    if (ctx == NULL)
        return NULL;

    ctx->un_skt_server_fd = -1;
    ctx->mcu_tty_fd       = -1;

    return ctx;
}

static void proxy_delete(proxy_ctx *ctx)
{
    close_fd(ctx->un_skt_server_fd);
    close_fd(ctx->mcu_tty_fd);

    if (ctx) {
        free(ctx);
    }
}

static void print_raw_log(void *raw_buff, uint32_t length)
{
    uint8_t *buff = (uint8_t *)raw_buff;
    for (uint32_t i = 0; i < length; ++i)
        printf("0x%02X ", buff[i]);
    printf("\n");
}

int32_t proxy_mcu_command(int32_t unix_cli_fd, int32_t mcu_tty_fd)
{
    struct pollfd pollfds[2] = {0};
    int32_t       rc         = -1;
    ssize_t       pkt_sz     = 0;

    socket_request_packet  req_pkt      = {0};
    mcu_packet             send_mcu_pkt = {0};
    mcu_packet            *recv_mcu_pkt = NULL;
    socket_response_packet resp_pkt     = {0};

    pollfds[0].fd     = unix_cli_fd;
    pollfds[0].events = POLLIN | POLLHUP | POLLERR;
    pollfds[1].fd     = mcu_tty_fd;
    pollfds[1].events = POLLIN | POLLOUT;

    log_debug("Start proxy MCU command");

    for (;;) {
        errno = 0;
        rc    = poll(pollfds, COUNT_OF_POLLFD(pollfds), 1000);
        if (rc < 0) {
            log_error("Poll failed");
            break;
        }

        if (pollfds[0].revents & (POLLHUP | POLLERR)) {
            log_error("unix socket error");
            break;
        }

        if (pollfds[0].revents & POLLIN) {
            errno = 0;
            rc    = read(unix_cli_fd, &req_pkt, sizeof(req_pkt));
            if (rc == 0) {
                log_warn("Unix client disconnected gracefully.");
                break;
            }
            else if (rc < 0 && (errno == ECONNRESET || errno == EPIPE)) {
                log_error("Unix disconnected forcefully.");
                break;
            }

            log_debug("Receive packet from unix socket client, rc: %d\n", rc);

            if (!is_mcu_command_support(req_pkt.command, current_model->info.mcu_features)) {
                resp_pkt.return_code     = ERR_CODE_COMMAND_NOT_SUPPORT;
                resp_pkt.payload.data_sz = 0;
                rc                       = write(unix_cli_fd, &resp_pkt, sizeof(socket_response_packet) + resp_pkt.payload.data_sz);
                log_warn("MCU command: %s is not support", req_pkt.command);
                break;
            }

            log_debug("MCU command is support: %s\n", req_pkt.command);

            pkt_sz = make_mcu_packet(&send_mcu_pkt, req_pkt.command, req_pkt.payload.data, req_pkt.payload.data_sz);

            log_debug("MCU command packet size: %zd\n", pkt_sz);

            errno = 0;
            rc    = mcu_write(&send_mcu_pkt, pkt_sz);
            if (rc == 0) {
                log_warn("TTY device disconnected gracefully.");
                break;
            }
            else if (rc < 0 && (errno == ECONNRESET || errno == EPIPE)) {
                log_error("TTY device  disconnected forcefully.");
                break;
            }

            log_debug("Send MCU command, rc: %d\n", rc);
        }

        if (pollfds[1].revents & POLLOUT) {
            /// TODO: Need to determine delay by command type'
            // When mcu is ready to write, delay
            log_debug("MCU device is ready to write. Delay");
            sleep_ms(DELAY_SEND_CMD);
        }

        if (pollfds[1].revents & POLLIN) {
            recv_mcu_pkt = (mcu_packet *)calloc(1, sizeof(mcu_packet) + MAX_DATA_SZ + sizeof(uint8_t));
            if (recv_mcu_pkt == NULL) {
                break;
            }

            errno = 0;
            rc    = mcu_read(recv_mcu_pkt, MAX_DATA_SZ);
            if (rc == 0) {
                log_warn("TTY device disconnected gracefully.");
                free(recv_mcu_pkt);
                break;
            }
            else if (rc < 0 && (errno == ECONNRESET || errno == EPIPE)) {
                log_error("TTY device disconnected forcefully.");
                free(recv_mcu_pkt);
                break;
            }

            log_debug("Receive packet from MCU socket server, rc: %d\n", rc);

#ifdef DEBUG
            print_raw_log(recv_mcu_pkt, pkt_sz);
#endif

            if (!validate_mcu_packet(recv_mcu_pkt)) {
                resp_pkt.return_code     = ERR_CODE_COMMAND_ILLEGAL_MCU_PKT;
                resp_pkt.payload.data_sz = 0;
            }
            else if (recv_mcu_pkt->attn == MX_MCU_NACK) {
                resp_pkt.return_code     = ERR_CODE_COMMAND_NACK_MCU_PKT;
                resp_pkt.payload.data_sz = 0;
            }
            else if (recv_mcu_pkt->attn == MX_MCU_ACK) {
                if (memcmp(recv_mcu_pkt->command, send_mcu_pkt.command, MX_MCU_CMD_SIZE)) {
                    resp_pkt.return_code     = ERR_CODE_COMMAND_UNEXPECTED_RESP;
                    resp_pkt.payload.data_sz = 0;
                }
                else {
                    memcpy(resp_pkt.payload.data, recv_mcu_pkt->data, recv_mcu_pkt->data_sz);
                    resp_pkt.return_code     = ERR_CODE_OK;
                    resp_pkt.payload.data_sz = recv_mcu_pkt->data_sz;

                    invoke_mcu_command_callback_fn(recv_mcu_pkt->command, recv_mcu_pkt->data);
                }
            }
            else {
                resp_pkt.return_code     = ERR_CODE_UNKNOWN;
                resp_pkt.payload.data_sz = 0;
            }
            free(recv_mcu_pkt);

            errno = 0;
            rc    = write(unix_cli_fd, &resp_pkt, sizeof(socket_response_packet) + resp_pkt.payload.data_sz);
            if (rc == 0) {
                log_warn("Unix client disconnected gracefully.");
                break;
            }
            else if (rc < 0 && (errno == ECONNRESET || errno == EPIPE)) {
                log_error("Unix disconnected forcefully.");
                break;
            }

            log_debug("Send Unix socket packet, rc: %d\n", rc);

            /// FIXME: Need to handle client disconnect
            break;
        }
    }

    log_debug("End proxy MCU command");
    return 0;
}

int32_t handle_unix_client(proxy_ctx *ctx)
{
    int32_t                unix_cli_fd = -1;
    socket_response_packet resp_pkt    = {0};
    int32_t                rc          = -1;

    for (;;) {
        errno       = 0;
        unix_cli_fd = accept(ctx->un_skt_server_fd, NULL, NULL);

        log_debug("Accept unix client, fd: %d\n", unix_cli_fd);

        if (is_valid_fd(unix_cli_fd)) {
            if (mcu_capability) {
                proxy_mcu_command(unix_cli_fd, ctx->mcu_tty_fd);
            }
            else {
                resp_pkt.return_code     = ERR_CODE_MCU_NOT_AVAILABLE;
                resp_pkt.payload.data_sz = 0;
                /// FIXME: make sure the fd is writeable, maybe use poll?
                rc = write(unix_cli_fd, &resp_pkt, sizeof(socket_response_packet) + resp_pkt.payload.data_sz);
                if (rc <= 0) {
                    log_warn("Response unix client failed.");
                }
            }

            close_fd(unix_cli_fd);
        }
        else {
            log_warn("Invaild unix client. %s(%d)", strerror(errno), errno);
        }
    }
}

void daemon_shutdown()
{
    log_info("Moxa MCU daemon shutdown.");
    proxy_delete(ctx);
    unlink(MX_MCU_DAEMON_UNIX_SOCK_PATH);
    logger_shutdown();
}

void handle_daemon_reload(int32_t signal)
{
    (void)signal;

    if (logger_get_level() != LOG_DEBUG) {
        log_info("Enable debug log");
        logger_set_level(LOG_DEBUG);
    }
    else {
        log_info("Disable debug log");
        logger_set_level(LOG_INFO);
    }
}

void handle_daemon_end(int32_t signal)
{
    (void)signal;

    log_info("Receive SIGINT or SIGTERM signal.");

    daemon_shutdown();
    exit(EXIT_SUCCESS);
}

bool setup_signals()
{
    int32_t          rc = 0;
    struct sigaction sa = {0};

    for (int s = SIGRTMAX; s; s--) {
        switch (s) {
        case SIGHUP:
            sa.sa_handler = handle_daemon_reload;
            // If accept() is interrupted by SIGHUP, then the call is automatically restarted after the handle_daemon_reload returns
            sa.sa_flags = SA_RESTART;
            break;
        case SIGINT:
        case SIGTERM:
            sa.sa_handler = handle_daemon_end;
            break;
        default:
            continue;
        }
        if ((rc = sigaction(s, &sa, NULL)) == -1)
            return false;
    }

    return true;
}

bool identify_model()
{
    char buffer[64];

    char *cmd1[] = {"/usr/sbin/dmidecode", "-t", "12", NULL};
    char *cmd2[] = {"grep", "Option", NULL};
    char *cmd3[] = {"awk", "-F", ":", "{print substr($2,1,11)}", NULL};
    char *cmd4[] = {"sed", "s/ //g", NULL};
    char *cmd5[] = {"tr", "-d", "'[:space:]'", NULL};

    char **commands[] = {cmd1, cmd2, cmd3, cmd4, cmd5};

    if (!execute_pipeline(commands, sizeof(commands) / sizeof(commands[0]), buffer)) {
        log_error("Error: get model name failed\n");
        return false;
    }

    for (int32_t i = 0; support_models[i].info.model_name != NULL; ++i) {
        if (!strncmp(buffer, support_models[i].info.model_name, support_models[i].info.name_prefix_sz)) {
            current_model = &support_models[i];
            return true;
        }
    }

    return false;
}

bool daemon_init()
{
    logger_init("mx-mcud");
    log_info("Moxa MCU daemon init.");

    if (!setup_signals()) {
        log_error("Setup daemon signals handler failed.");
    }

    if (!identify_model()) {
        log_error("Unknown model.");
        return false;
    }

    log_info("The model %s is support.", current_model->info.model_name);

    current_model->init_func(&current_model->info);

    ctx = proxy_new();
    if (ctx == NULL) {
        return false;
    }

    ctx->un_skt_server_fd = create_unix_skt_server(MX_MCU_DAEMON_UNIX_SOCK_PATH);
    ctx->mcu_tty_fd       = mcu_device_open(current_model->info.mcu_tty_device);
    if (ctx->mcu_tty_fd < 0) {
        return false;
    }

    return true;
}

int32_t main(void)
{

    if (!daemon_init()) {
        log_error("MCU daemon initialization failed.");
        exit(EXIT_FAILURE);
    }

    handle_unix_client(ctx);

    daemon_shutdown();

    return 0;
}