/*
 * Copyright (C) MOXA Inc. All rights reserved.
 * Authors:
 *     2024  Wilson YS Huang  <wilsonys.huang@moxa.com>
 * This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
 * See the file LICENSE for details.
 */

#ifndef _MCU_H_
#define _MCU_H_

#include <fcntl.h>
#include <pthread.h>
#include <stdbool.h>
#include <stddef.h>
#include <stdint.h>
#include <stdlib.h>
#include <string.h>
#include <sys/file.h>
#include <sys/poll.h>
#include <termios.h>
#include <unistd.h>

#include <common.h>

#define MX_MCU_DAEMON_UNIX_SOCK_PATH "/var/run/mx-mcud.sock"

#define MX_MCU_CMD_SIZE 3
#define MX_MCU_CMD      0x07   // Command Packet
#define MX_MCU_ACK      0x06   // Acknowledge Packet
#define MX_MCU_NACK     0x15   // Negative ACK Packet
// 0x15 0x07 0xFF 0x52 0x45 0x00 0x4D 0x15 0xFF 0x52 0x45 0x4C 0x00 0x08
// 0x15 0xFF 0x52 0x45 0x07 0x00 0x4D 0x15 0xFF 0x52 0x45 0x4C 0x00 0x08
#define MX_MCU_ADDR_BROADCAST 0xFF

#define MX_MCU_VERSION_CMD             "\x53\x57\x49"
#define MX_MCU_RELAY_MODE_CMD          "\x52\x45\x4C"
#define MX_MCU_WDT_RESET_MODE_CMD      "\x57\x52\x53"
#define MX_MCU_WDT_RELAY_MODE_CMD      "\x57\x52\x4C"
#define MX_MCU_POWEROFF_RELAY_MODE_CMD "\x50\x52\x4C"
#define MX_MCU_APP_WDT_RELAY_MODE_CMD  "\x41\x57\x4C"
#define MX_MCU_APP_WDT_RESET_MODE_CMD  "\x41\x57\x52"
#define MX_MCU_APP_WDT_TIMEOUT_CMD     "\x41\x57\x54"
#define MX_MCU_APP_TICK_CMD            MX_MCU_APP_WDT_TIMEOUT_CMD

#define FEAT_MCU_VERSION                BIT(0)
#define FEAT_MCU_RELAY_MODE             BIT(1)
#define FEAT_MCU_WDT_RESET_MODE         BIT(2)
#define FEAT_MCU_WDT_RELAY_MODE         BIT(3)
#define FEAT_MCU_POWEROFF_RELAY_MODE    BIT(4)
#define FEAT_MCU_APP_WDT_RELAY_MODE     BIT(5)
#define FEAT_MCU_APP_WDT_RESET_MODE_CMD BIT(6)
#define FEAT_MCU_APP_WDT_TIMEOUT_CMD    BIT(7)

#define DELAY_START_UPGRADE 300
#define DELAY_SEND_PKT      200
#define DELAY_SEND_CMD      100

typedef struct {
    uint8_t attn;
    uint8_t addr;
    uint8_t command[3];
    uint8_t data_sz;
    uint8_t hdr_chk;
    uint8_t data[];
} __attribute__((packed)) mcu_packet;

bool    is_mcu_command_support(uint8_t *command, int8_t features);
size_t  make_mcu_packet(mcu_packet *mcu_pkt, uint8_t command[3], uint8_t *data, size_t data_sz);
bool    validate_mcu_packet(mcu_packet *pkt);
bool    invoke_mcu_command_callback_fn(uint8_t *command, void *data);
int32_t mcu_device_open(const char *tty_name);
ssize_t mcu_read(void *buf, size_t nbytes);
ssize_t mcu_write(void *buf, size_t nbytes);

#endif