#!/bin/bash
#
# Copyright (C) 2024 MOXA Inc. All rights reserved.
# This software is distributed under the terms of the MOXA SOFTWARE NOTICE.
# See the file LICENSE for details.
#
# Authors:
#       2025  Elvis CW Yao  <elviscw.yao@moxa.com>

# Format of dry-run list:
# --------------------------------------------------
#  Product Name: BXPC100
#  Architecture: x86_64
#  Kernel Version:6.1.0-18-amd64
#  UEFI Secure Boot:disabled
# --------------------------------------------------
#  Name                          Version
# ==================================================
#  moxa-it87-gpio-driver         5.2+1.4.0-1
#  moxa-it87-wdt-driver          5.2+1.5.0
#  moxa-it87-serial-driver       1.4.1+u2
#  moxa-mxuport-driver           6.1.1_build_24061315
#  moxa-x86-control-tools        1.8.1
#  libgpiod                      2.0+1.0.0
# --------------------------------------------------

if [ $# -ne 3 ]; then
        echo "Usage: ${0} <old_list> <new_list> <src_dir>" >&2
        exit 2
fi

OLD_LIST=${1}
NEW_LIST=${2}
SRC_DIR=${3}
if [ ! -f "${OLD_LIST}" ]; then
        echo "Error: old list not exist" >&2
        exit 1
fi
if [ ! -f "${NEW_LIST}" ]; then
        echo "Error: new list not exist" >&2
        exit 1
fi
if [ ! -d "${SRC_DIR}" ]; then
        echo "Error: source dir not exist" >&2
        exit 1
fi

OUTPUT_CHANGELOG="${OLD_LIST}_${NEW_LIST}_changelog"
in_table=0

echo "===============================" >"${OUTPUT_CHANGELOG}"
while IFS= read -r line; do
        # Check for the table start marker (e.g., a line of '=')
        if [[ "$line" =~ ^=+ ]]; then
                in_table=1
                continue
        fi

        # Check for the table end marker (e.g., a line of '-' after the table)
        if [[ "$line" =~ ^-+ ]]; then
                in_table=0
                continue
        fi

        # If inside the table and the line is not empty
        if [[ $in_table -eq 1 && "$line" =~ [a-zA-Z0-9] ]]; then
                # Extract name and version
                name=$(echo "$line" | awk '{print $1}')
                new_version=$(echo "$line" | awk '{print $2}')

                if old_version=$(grep -m1 "$name" "${OLD_LIST}" | awk -F' ' '{print $2}'); then
                        if [ "$new_version" != "$old_version" ]; then
                                echo "[$name] changelog" >>"${OUTPUT_CHANGELOG}"
                                if [ -z "$old_version" ]; then
                                        sed -n "/^## \[${new_version}\]/,\$p" \
                                                "${SRC_DIR}/${name}-${new_version}/CHANGELOG" >>"${OUTPUT_CHANGELOG}"
                                else
                                        sed -n "/## \[${new_version}\]/,/## \[${old_version}/p" \
                                                "${SRC_DIR}/${name}-${new_version}/CHANGELOG" | head -n -1 >>"${OUTPUT_CHANGELOG}"
                                fi
                                echo "===============================" >>"${OUTPUT_CHANGELOG}"
                        fi
                else
                        continue
                fi
        fi
done <"${NEW_LIST}"

echo "Changelog output file: '${OUTPUT_CHANGELOG}'"
exit 0
